/*---------------------------------------------------------------------------*\

    FILE....: TSAMECHO.CPP
    TYPE....: C++ program
    AUTHOR..: David Rowe
    DATE....: 9/2/01

    Test program for sampling echo from V4PCI card hybrid, used to adjust
    the TS5070 differential amp for V4PCI

    Compile: gcc tsamecho.cpp -lvpb -g -Wall -o tsamecho -lm -pthread

    Instructions:
    1. Connect line to port 1, and start program.
    2. Dial both port 1.
    3. Input signal will be driven onto line and ref will be recorded.
 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "../src/mess.h"
#include "../src/comm.h"
#include "../src/config.h"
#include "../src/dspfifo.h"
#include "../src/timer.h"
#include "../src/timer.h"
#include "../src/wobbly.h"
#include "../src/wavecpp.h"
#include "../src/vpbapi.h"
#include "../src/mapdev.h"
#include "../src/translate.h"
extern "C" {
#include "../src/alawmulaw.h"
}

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

int kbhit();

#define WAIT		1	// time out delay 
#define	BOARD		0	// board number
#define	N	        160	// size of processing frame
#define LOG_FRAMES      200     // number of frames to log

word offhook[] = {
	PC_LCODEC_OFFHK,
	PC_CODEC_OFFHK,
	0			// channel
};

word onhook[] = {
	PC_LCODEC_ONHK,
	PC_CODEC_ONHK,
	0			// channel
};

void set_codec_reg(int    chdev,
	           unsigned short addr, 
          	   unsigned short data,  
		   Comm   *c
		  );

void play_set_hw_gain(int handle, float gain, Comm *c);
void record_set_hw_gain(int handle, float gain, Comm *c);

int main(int argc, char *argv[])
{
	long		i;
	VPBREG		*v;
	word		upmess[COMM_MAX_MESS];
	char		s[MAX_STR];
	void		*win,*wref;
	int		ch1_offhook;
	short		buf[N];
	char            charbuf[N];
	int		frames, rd;

	if (argc != 3) {
		printf("usage: %s InputFile(D/A) ReferenceFile(A/D)\n",argv[0]);
		exit(0);
	}

	wave_open_read(&win, argv[1]);
	wave_open_write(&wref, argv[2], VPB_LINEAR);

	// initialise 

	try {
		Timer	timer;
		mess_mprintf_on();

		Comm comm;
		v = comm.vpbreg(BOARD);

		// configure VPB signal processing --------------------------

		config_create_object(&comm, BOARD, FIFO_DOWN, 0, 0, 
				     v->szrxdf[0]);
		config_create_object(&comm, BOARD, LIN2ALAW , 1, 0, 0);
		config_create_object(&comm, BOARD, CODEC_DA , 2, 0, 0);
		config_create_object(&comm, BOARD, CODEC_AD , 3, 0, 0);
		config_create_object(&comm, BOARD, ALAW2LIN , 4, 0, 0);
		config_create_object(&comm, BOARD, FIFO_UP  , 5, 1, 
				     v->szrxdf[1]);

		config_create_wire(&comm, BOARD, 0, 1);
		config_create_wire(&comm, BOARD, 1, 2);
		config_create_wire(&comm, BOARD, 3, 4);
		config_create_wire(&comm, BOARD, 4, 5);

		onhook[2] = 0;
		comm.PutMessageVPB(BOARD,onhook);
		ch1_offhook = 0;

		// make sure tx & rx gains 0dB
		play_set_hw_gain(0, -1, &comm);
		record_set_hw_gain(0, 12, &comm);

		// default set codec diff amp (V4PCI only)
		int reg = 0xc7;
		set_codec_reg(0, 0x32, reg, &comm);
		set_codec_reg(0, 0x42, 0x00, &comm);
		printf("codec reg set to 0x%x, def is 0xc7\n", reg);

		// start main processing loop ------------------------------
		
		config_run(&comm, BOARD);
		i = 0;
		frames = 0;
		printf("waiting for ring on port 0 (1)\n");
		rd = 1;

		do {
			// check for events from VPB

			while(comm.GetMessageVPB(BOARD, upmess) == OK) {
				translate(BOARD, s, upmess);
				mprintf(s);
				if (upmess[1] == DSP_CODEC_RING) {
					offhook[2] = upmess[2];
					comm.PutMessageVPB(BOARD,offhook);
					if (upmess[2] == 0) {
						vpb_sleep(1000);
						ch1_offhook = 1;
					}
						
				}
			}


			// empty Rx FIFO
			while(v->rxdf[1]->Read((word*)buf, N) == OK);
			if (ch1_offhook) {

				rd = vpb_wave_read(win,(char*)buf,
						   sizeof(short)*N);
				// tmp fix for dsp alaw clipping bug
				alaw_encode(charbuf, buf, N);
				alaw_decode(buf, charbuf, N);
				while(v->txdf[0]->Write((word*)buf, N) != OK) {
					comm.CheckForAssert(BOARD);
					vpb_sleep(1);
				}
				while(v->rxdf[1]->Read((word*)buf, N) != OK) {
					comm.CheckForAssert(BOARD);
					vpb_sleep(1);
				}

				// time align tx ref wave file to input
				if (frames > 1) {
					vpb_wave_write(wref, (char*)buf, 
						       sizeof(buf));
				}
				
				printf("\rframes: %d",frames++);

			}

		} while(rd);

		// clean up and finish 

		onhook[2] = 0;
		comm.PutMessageVPB(BOARD,onhook);
	        ch1_offhook = 0;
		vpb_sleep(300);

		// send clear config message -------------------------------

		config_stop(&comm, BOARD);
		config_clear(&comm, BOARD);

		vpb_wave_close_read(win);
		vpb_wave_close_write(wref);

	}	// try ...

	catch (Wobbly w) {
		char	s[MAX_STR];
		
		w.translate(s);
		
		if (w.file[0] != 0) 
			printf("exception caught: %s, %s, line = %d\n",s, 
			       w.file, w.line);
		else
			printf("exception caught: %s\n",s);

		printf("Press any key to exit....\n");
	}

	return 0;
}

